/**
 * GreenAdmin - Admin Panel JavaScript
 * Supports: 3-level sidebar, desktop collapse, theme toggle, dropdowns, FAQ, DataTable
 */
(function () {
  'use strict';

  // Sidebar Manager (mobile + desktop collapse + 3-level submenu)
  const SidebarManager = {
    storageKey: 'ga-sidebar-collapsed',

    init() {
      this.sidebar = document.querySelector('.ga-sidebar');
      this.overlay = document.querySelector('.ga-overlay');
      this.hamburger = document.querySelector('.ga-hamburger');
      this.isMobile = () => window.innerWidth <= 1024;

      // Restore desktop collapsed state
      if (!this.isMobile() && localStorage.getItem(this.storageKey) === '1') {
        document.body.classList.add('sidebar-collapsed');
      }

      if (this.hamburger) {
        this.hamburger.addEventListener('click', () => {
          if (this.isMobile()) {
            this.mobileToggle();
          } else {
            this.desktopToggle();
          }
        });
      }

      if (this.overlay) {
        this.overlay.addEventListener('click', () => this.mobileClose());
      }

      // Level 2 submenu toggles
      document.querySelectorAll('.ga-nav-toggle').forEach(item => {
        item.addEventListener('click', (e) => {
          e.preventDefault();
          const submenu = item.nextElementSibling;
          if (submenu && submenu.classList.contains('ga-submenu')) {
            item.classList.toggle('expanded');
            submenu.classList.toggle('open');
          }
        });
      });

      // Level 3 submenu toggles
      document.querySelectorAll('.ga-nav-toggle-l3').forEach(item => {
        item.addEventListener('click', (e) => {
          e.preventDefault();
          const submenu = item.nextElementSibling;
          if (submenu && submenu.classList.contains('ga-submenu-l3')) {
            item.classList.toggle('expanded');
            submenu.classList.toggle('open');
          }
        });
      });

      // Auto-expand active submenu on load + mark parent as has-active
      document.querySelectorAll('.ga-submenu .ga-nav-item.active, .ga-submenu-l3 .ga-nav-item.active').forEach(activeItem => {
        const l3 = activeItem.closest('.ga-submenu-l3');
        if (l3) {
          l3.classList.add('open');
          const l3Toggle = l3.previousElementSibling;
          if (l3Toggle) l3Toggle.classList.add('expanded');
        }
        const l2 = activeItem.closest('.ga-submenu');
        if (l2) {
          l2.classList.add('open');
          const l2Toggle = l2.previousElementSibling;
          if (l2Toggle) {
            l2Toggle.classList.add('expanded');
            l2Toggle.classList.add('has-active');
          }
        }
      });
    },

    mobileToggle() {
      if (this.sidebar) {
        this.sidebar.classList.toggle('open');
        if (this.overlay) this.overlay.classList.toggle('show');
      }
    },

    mobileClose() {
      if (this.sidebar) {
        this.sidebar.classList.remove('open');
        if (this.overlay) this.overlay.classList.remove('show');
      }
    },

    desktopToggle() {
      document.body.classList.toggle('sidebar-collapsed');
      const collapsed = document.body.classList.contains('sidebar-collapsed');
      localStorage.setItem(this.storageKey, collapsed ? '1' : '0');
    }
  };

  // Theme Manager (mode + color + panel)
  const ThemeManager = {
    modeKey: 'ga-theme-mode',
    colorKey: 'ga-theme-color',
    modes: ['light', 'dark', 'mix'],
    colors: ['green', 'blue', 'purple', 'red', 'orange', 'teal', 'indigo', 'rose', 'amber', 'cyan'],
    defaultMode: 'mix',
    defaultColor: 'green',

    init() {
      this.btn = document.getElementById('themeToggle');
      // Migrate old localStorage key
      const legacyTheme = localStorage.getItem('ga-theme');
      if (legacyTheme && !localStorage.getItem(this.modeKey)) {
        const mapped = (legacyTheme === 'dark') ? 'dark' : (legacyTheme === 'light') ? 'mix' : legacyTheme;
        localStorage.setItem(this.modeKey, mapped);
        localStorage.removeItem('ga-theme');
      }
      const savedMode = localStorage.getItem(this.modeKey) || this.defaultMode;
      const savedColor = localStorage.getItem(this.colorKey) || this.defaultColor;
      this.applyMode(savedMode, false);
      this.applyColor(savedColor, false);
      this.updateIcon();
      if (this.btn) {
        this.btn.addEventListener('click', () => this.openPanel());
      }
      this.injectPanel();
    },

    applyMode(mode, animate) {
      if (animate) {
        document.documentElement.classList.add('ga-theme-transitioning');
        setTimeout(() => document.documentElement.classList.remove('ga-theme-transitioning'), 400);
      }
      document.documentElement.setAttribute('data-theme', mode);
      localStorage.setItem(this.modeKey, mode);
      this.updateIcon();
      this.updatePanel();
    },

    applyColor(color, animate) {
      if (animate) {
        document.documentElement.classList.add('ga-theme-transitioning');
        setTimeout(() => document.documentElement.classList.remove('ga-theme-transitioning'), 400);
      }
      document.documentElement.setAttribute('data-color', color);
      localStorage.setItem(this.colorKey, color);
      this.updatePanel();
    },

    getMode() {
      return document.documentElement.getAttribute('data-theme') || this.defaultMode;
    },

    getColor() {
      return document.documentElement.getAttribute('data-color') || this.defaultColor;
    },

    cycleModes() {
      const current = this.getMode();
      const idx = this.modes.indexOf(current);
      const next = this.modes[(idx + 1) % this.modes.length];
      this.applyMode(next, true);
    },

    reset() {
      this.applyMode(this.defaultMode, true);
      this.applyColor(this.defaultColor, true);
    },

    updateIcon() {
      if (!this.btn) return;
      const mode = this.getMode();
      const icons = {
        light: '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="5"/><path d="M12 1v2M12 21v2M4.22 4.22l1.42 1.42M18.36 18.36l1.42 1.42M1 12h2M21 12h2M4.22 19.78l1.42-1.42M18.36 5.64l1.42-1.42"/></svg>',
        dark: '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M21 12.79A9 9 0 1111.21 3 7 7 0 0021 12.79z"/></svg>',
        mix: '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"/><path d="M12 2a10 10 0 0 1 0 20V2z" fill="currentColor" opacity="0.2"/><path d="M12 2v20"/></svg>'
      };
      this.btn.innerHTML = icons[mode] || icons.mix;
    },

    openPanel() {
      const panel = document.querySelector('.ga-theme-panel');
      const overlay = document.querySelector('.ga-theme-overlay');
      if (panel) panel.classList.add('open');
      if (overlay) overlay.classList.add('open');
    },

    closePanel() {
      const panel = document.querySelector('.ga-theme-panel');
      const overlay = document.querySelector('.ga-theme-overlay');
      if (panel) panel.classList.remove('open');
      if (overlay) overlay.classList.remove('open');
    },

    updatePanel() {
      const mode = this.getMode();
      const color = this.getColor();
      document.querySelectorAll('.ga-mode-card').forEach(card => {
        card.classList.toggle('active', card.dataset.mode === mode);
      });
      document.querySelectorAll('.ga-color-swatch').forEach(swatch => {
        swatch.classList.toggle('active', swatch.dataset.color === color);
      });
      const info = document.querySelector('.ga-theme-current-text');
      if (info) {
        const modeLabels = { light: 'Light', dark: 'Dark', mix: 'Mix' };
        info.innerHTML = '<strong>' + (modeLabels[mode] || 'Mix') + '</strong> mode &middot; <strong>' + color + '</strong> color';
      }
    },

    injectPanel() {
      if (document.querySelector('.ga-theme-panel')) return;

      const colorItems = this.colors.map(c =>
        '<div class="ga-color-swatch-item">' +
        '<div class="ga-color-swatch" data-color="' + c + '" title="' + c + '"></div>' +
        '<span class="ga-color-swatch-label">' + c + '</span>' +
        '</div>'
      ).join('');

      const html = '' +
        '<button class="ga-theme-toggle-fab" title="Theme Settings">' +
          '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="3"/><path d="M19.4 15a1.65 1.65 0 0 0 .33 1.82l.06.06a2 2 0 0 1-2.83 2.83l-.06-.06a1.65 1.65 0 0 0-1.82-.33 1.65 1.65 0 0 0-1 1.51V21a2 2 0 0 1-4 0v-.09A1.65 1.65 0 0 0 9 19.4a1.65 1.65 0 0 0-1.82.33l-.06.06a2 2 0 0 1-2.83-2.83l.06-.06A1.65 1.65 0 0 0 4.68 15a1.65 1.65 0 0 0-1.51-1H3a2 2 0 0 1 0-4h.09A1.65 1.65 0 0 0 4.6 9a1.65 1.65 0 0 0-.33-1.82l-.06-.06a2 2 0 0 1 2.83-2.83l.06.06A1.65 1.65 0 0 0 9 4.68a1.65 1.65 0 0 0 1-1.51V3a2 2 0 0 1 4 0v.09a1.65 1.65 0 0 0 1 1.51 1.65 1.65 0 0 0 1.82-.33l.06-.06a2 2 0 0 1 2.83 2.83l-.06.06A1.65 1.65 0 0 0 19.4 9a1.65 1.65 0 0 0 1.51 1H21a2 2 0 0 1 0 4h-.09a1.65 1.65 0 0 0-1.51 1z"/></svg>' +
        '</button>' +
        '<div class="ga-theme-overlay"></div>' +
        '<div class="ga-theme-panel">' +
          '<div class="ga-theme-panel-header">' +
            '<div class="ga-theme-panel-title">' +
              '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M12 2.69l5.66 5.66a8 8 0 1 1-11.31 0z"/></svg>' +
              'Theme Customizer' +
            '</div>' +
            '<button class="ga-theme-panel-close">' +
              '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><line x1="18" y1="6" x2="6" y2="18"/><line x1="6" y1="6" x2="18" y2="18"/></svg>' +
            '</button>' +
          '</div>' +
          '<div class="ga-theme-panel-body">' +
            '<div class="ga-theme-current">' +
              '<div class="ga-theme-current-dot"></div>' +
              '<div class="ga-theme-current-text"></div>' +
            '</div>' +
            '<div class="ga-theme-section">' +
              '<div class="ga-theme-section-label">' +
                '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="5"/><path d="M12 1v2M12 21v2M4.22 4.22l1.42 1.42M18.36 18.36l1.42 1.42M1 12h2M21 12h2M4.22 19.78l1.42-1.42M18.36 5.64l1.42-1.42"/></svg>' +
                'Appearance Mode' +
              '</div>' +
              '<div class="ga-mode-grid">' +
                '<div class="ga-mode-card" data-mode="light">' +
                  '<div class="ga-mode-preview"><div class="ga-mode-preview-sidebar"></div><div class="ga-mode-preview-content"><div class="ga-mode-preview-header"></div><div class="ga-mode-preview-body"></div></div></div>' +
                  '<div class="ga-mode-label">Light</div>' +
                '</div>' +
                '<div class="ga-mode-card" data-mode="dark">' +
                  '<div class="ga-mode-preview"><div class="ga-mode-preview-sidebar"></div><div class="ga-mode-preview-content"><div class="ga-mode-preview-header"></div><div class="ga-mode-preview-body"></div></div></div>' +
                  '<div class="ga-mode-label">Dark</div>' +
                '</div>' +
                '<div class="ga-mode-card" data-mode="mix">' +
                  '<div class="ga-mode-preview"><div class="ga-mode-preview-sidebar"></div><div class="ga-mode-preview-content"><div class="ga-mode-preview-header"></div><div class="ga-mode-preview-body"></div></div></div>' +
                  '<div class="ga-mode-label">Mix</div>' +
                '</div>' +
              '</div>' +
            '</div>' +
            '<div class="ga-theme-section">' +
              '<div class="ga-theme-section-label">' +
                '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M12 2.69l5.66 5.66a8 8 0 1 1-11.31 0z"/></svg>' +
                'Color Palette' +
              '</div>' +
              '<div class="ga-color-grid">' + colorItems + '</div>' +
            '</div>' +
          '</div>' +
          '<div class="ga-theme-panel-footer">' +
            '<button class="ga-theme-reset-btn">' +
              '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="1 4 1 10 7 10"/><path d="M3.51 15a9 9 0 1 0 2.13-9.36L1 10"/></svg>' +
              'Reset to Default' +
            '</button>' +
          '</div>' +
        '</div>';

      const container = document.createElement('div');
      container.id = 'ga-theme-system';
      container.innerHTML = html;
      document.body.appendChild(container);

      // Bind events
      const fab = container.querySelector('.ga-theme-toggle-fab');
      const overlay = container.querySelector('.ga-theme-overlay');
      const closeBtn = container.querySelector('.ga-theme-panel-close');
      const resetBtn = container.querySelector('.ga-theme-reset-btn');

      if (fab) fab.addEventListener('click', () => this.openPanel());
      if (overlay) overlay.addEventListener('click', () => this.closePanel());
      if (closeBtn) closeBtn.addEventListener('click', () => this.closePanel());
      if (resetBtn) resetBtn.addEventListener('click', () => this.reset());

      container.querySelectorAll('.ga-mode-card').forEach(card => {
        card.addEventListener('click', () => this.applyMode(card.dataset.mode, true));
      });

      container.querySelectorAll('.ga-color-swatch').forEach(swatch => {
        swatch.addEventListener('click', () => this.applyColor(swatch.dataset.color, true));
      });

      // Keyboard shortcut: Escape to close
      document.addEventListener('keydown', (e) => {
        if (e.key === 'Escape') this.closePanel();
      });

      this.updatePanel();
    }
  };

  // Dropdown
  const DropdownManager = {
    init() {
      document.addEventListener('click', (e) => {
        const trigger = e.target.closest('[data-dropdown]');
        if (trigger) {
          e.stopPropagation();
          const menu = trigger.nextElementSibling;
          if (menu) {
            document.querySelectorAll('.ga-dropdown-menu.show').forEach(m => {
              if (m !== menu) m.classList.remove('show');
            });
            menu.classList.toggle('show');
          }
          return;
        }
        document.querySelectorAll('.ga-dropdown-menu.show').forEach(m => m.classList.remove('show'));
      });
    }
  };

  // Header Menu Search (live suggestions)
  const HeaderSearchManager = {
    init() {
      this.container = document.querySelector('.ga-header-search');
      this.input = this.container ? this.container.querySelector('input') : null;
      if (!this.container || !this.input) return;

      this.results = document.createElement('div');
      this.results.className = 'ga-header-search-results';
      this.results.setAttribute('role', 'listbox');
      this.container.appendChild(this.results);

      this.sidebar = document.querySelector('.ga-sidebar');
      this.items = [];
      this.matches = [];
      this.activeIndex = -1;

      this.refreshItems();
      this.observeMenuChanges();

      this.input.setAttribute('autocomplete', 'off');
      this.input.setAttribute('placeholder', 'Search menu...');

      this.input.addEventListener('input', () => this.onInput());
      this.input.addEventListener('focus', () => this.onInput());
      this.input.addEventListener('keydown', (e) => this.onKeyDown(e));

      document.addEventListener('click', (e) => {
        if (!this.container.contains(e.target)) {
          this.hide();
        }
      });
    },

    collectItems() {
      const items = [];
      const seen = new Set();

      const addItem = (item) => {
        if (!item.label || !item.href) return;
        const key = `${item.label.toLowerCase()}|${item.href.toLowerCase()}`;
        if (seen.has(key)) return;
        seen.add(key);

        const keywords = Array.isArray(item.keywords) ? item.keywords.join(' ') : '';
        const hrefText = item.href.replace(/[-_/.]+/g, ' ');
        items.push({
          label: item.label,
          href: item.href,
          section: item.section || 'Menu',
          searchText: `${item.label} ${item.section || ''} ${keywords} ${item.href} ${hrefText}`.toLowerCase()
        });
      };

      const links = this.sidebar ? this.sidebar.querySelectorAll('a.ga-nav-item[href]') : [];

      links.forEach((link) => {
        const label = this.getLabel(link);
        const href = (link.getAttribute('href') || '').trim();
        if (!href || href === '#' || href.startsWith('javascript:')) return;
        const section = this.getSection(link);
        addItem({ label, href, section });
      });

      return items;
    },

    refreshItems() {
      this.items = this.collectItems();
    },

    observeMenuChanges() {
      if (!this.sidebar || typeof MutationObserver === 'undefined') return;

      this.observer = new MutationObserver(() => {
        this.refreshItems();
        if (this.input.value.trim()) {
          this.onInput();
        }
      });

      this.observer.observe(this.sidebar, {
        childList: true,
        subtree: true,
        characterData: true,
        attributes: true,
        attributeFilter: ['href', 'class']
      });
    },

    getLabel(link) {
      const span = link.querySelector('span');
      if (span && span.textContent.trim()) return span.textContent.trim();
      return link.textContent.trim();
    },

    getSection(link) {
      const group = link.closest('.ga-nav-group');
      const groupLabel = group ? (group.querySelector('.ga-nav-label')?.textContent || '').trim() : '';
      const trail = [];

      if (groupLabel) trail.push(groupLabel);

      const l2 = link.closest('.ga-submenu');
      if (l2) {
        const l2Toggle = l2.previousElementSibling;
        const l2Label = this.getToggleLabel(l2Toggle);
        if (l2Label) trail.push(l2Label);
      }

      const l3 = link.closest('.ga-submenu-l3');
      if (l3) {
        const l3Toggle = l3.previousElementSibling;
        const l3Label = this.getToggleLabel(l3Toggle);
        if (l3Label) trail.push(l3Label);
      }

      return trail.join(' / ') || 'Menu';
    },

    getToggleLabel(toggle) {
      if (!toggle) return '';
      const span = toggle.querySelector('span');
      return span ? span.textContent.trim() : toggle.textContent.trim();
    },

    onInput() {
      const query = this.input.value.toLowerCase().trim();
      if (!query) {
        this.hide();
        return;
      }

      this.refreshItems();
      this.matches = this.items
        .filter((item) => item.searchText.includes(query))
        .sort((a, b) => this.rank(a, query) - this.rank(b, query) || a.label.localeCompare(b.label))
        .slice(0, 8);

      this.activeIndex = -1;
      this.render();
    },

    rank(item, query) {
      const label = item.label.toLowerCase();
      const section = item.section.toLowerCase();
      if (label.startsWith(query)) return 0;
      if (label.includes(query)) return 1;
      if (section.includes(query)) return 2;
      return 3;
    },

    onKeyDown(e) {
      if (!this.matches.length) return;

      if (e.key === 'ArrowDown') {
        e.preventDefault();
        this.activeIndex = (this.activeIndex + 1) % this.matches.length;
        this.render();
      } else if (e.key === 'ArrowUp') {
        e.preventDefault();
        this.activeIndex = this.activeIndex <= 0 ? this.matches.length - 1 : this.activeIndex - 1;
        this.render();
      } else if (e.key === 'Enter') {
        const selected = this.matches[this.activeIndex] || this.matches[0];
        if (selected) {
          e.preventDefault();
          window.location.href = selected.href;
        }
      } else if (e.key === 'Escape') {
        this.hide();
      }
    },

    render() {
      if (!this.matches.length) {
        this.hide();
        return;
      }

      this.results.innerHTML = this.matches.map((item, index) => `
        <button type="button" class="ga-header-search-item ${index === this.activeIndex ? 'active' : ''}" data-index="${index}">
          <span class="ga-header-search-title">${item.label}</span>
          <span class="ga-header-search-meta">${item.section}</span>
        </button>
      `).join('');

      this.results.querySelectorAll('.ga-header-search-item').forEach((btn) => {
        btn.addEventListener('mousedown', (e) => e.preventDefault());
        btn.addEventListener('click', () => {
          const index = parseInt(btn.dataset.index, 10);
          const selected = this.matches[index];
          if (selected) window.location.href = selected.href;
        });
      });

      this.results.classList.add('show');
    },

    hide() {
      this.results.classList.remove('show');
      this.activeIndex = -1;
    }
  };

  // FAQ Accordion
  const FAQManager = {
    init() {
      document.querySelectorAll('.ga-faq-question').forEach(q => {
        q.addEventListener('click', () => {
          const item = q.parentElement;
          const wasOpen = item.classList.contains('open');
          // Close all
          document.querySelectorAll('.ga-faq-item.open').forEach(i => i.classList.remove('open'));
          if (!wasOpen) item.classList.add('open');
        });
      });
    }
  };

  // Alert dismissals
  const AlertManager = {
    init() {
      document.querySelectorAll('.ga-alert-close').forEach(btn => {
        btn.addEventListener('click', () => {
          const alert = btn.closest('.ga-alert');
          if (alert) {
            alert.style.opacity = '0';
            alert.style.transform = 'translateX(20px)';
            setTimeout(() => alert.remove(), 200);
          }
        });
      });
    }
  };

  // DataTable Engine
  window.GaDataTable = class {
    constructor(tableId, options = {}) {
      this.table = document.getElementById(tableId);
      if (!this.table) return;

      this.wrapper = this.table.closest('.ga-card');
      this.tbody = this.table.querySelector('tbody');
      this.allRows = Array.from(this.tbody.querySelectorAll('tr'));
      this.filteredRows = [...this.allRows];
      this.perPage = options.perPage || 10;
      this.currentPage = 1;
      this.sortCol = -1;
      this.sortDir = 'asc';

      this.initControls();
      this.initSorting();
      this.render();
    }

    initControls() {
      const controls = this.wrapper.querySelector('.ga-dt-controls');
      if (!controls) return;

      this.searchInput = controls.querySelector('.ga-dt-search input');
      this.perPageSelect = controls.querySelector('.ga-dt-per-page select');

      if (this.searchInput) {
        this.searchInput.addEventListener('input', () => {
          this.currentPage = 1;
          this.applyFilter();
          this.render();
        });
      }

      if (this.perPageSelect) {
        this.perPageSelect.addEventListener('change', () => {
          this.perPage = parseInt(this.perPageSelect.value);
          this.currentPage = 1;
          this.render();
        });
      }
    }

    initSorting() {
      this.table.querySelectorAll('thead th.sortable').forEach((th, i) => {
        th.addEventListener('click', () => {
          if (this.sortCol === i) {
            this.sortDir = this.sortDir === 'asc' ? 'desc' : 'asc';
          } else {
            this.sortCol = i;
            this.sortDir = 'asc';
          }
          this.table.querySelectorAll('thead th.sortable').forEach(h => h.classList.remove('asc', 'desc'));
          th.classList.add(this.sortDir);
          this.applySort();
          this.render();
        });
      });
    }

    applyFilter() {
      const q = (this.searchInput ? this.searchInput.value : '').toLowerCase().trim();
      this.filteredRows = this.allRows.filter(row => {
        return q === '' || row.textContent.toLowerCase().includes(q);
      });
      if (this.sortCol >= 0) this.applySort();
    }

    applySort() {
      const col = this.sortCol;
      const dir = this.sortDir === 'asc' ? 1 : -1;
      this.filteredRows.sort((a, b) => {
        const aVal = (a.cells[col]?.textContent || '').trim();
        const bVal = (b.cells[col]?.textContent || '').trim();
        const aNum = parseFloat(aVal.replace(/[^0-9.-]/g, ''));
        const bNum = parseFloat(bVal.replace(/[^0-9.-]/g, ''));
        if (!isNaN(aNum) && !isNaN(bNum)) return (aNum - bNum) * dir;
        return aVal.localeCompare(bVal) * dir;
      });
    }

    render() {
      const total = this.filteredRows.length;
      const totalPages = Math.max(1, Math.ceil(total / this.perPage));
      if (this.currentPage > totalPages) this.currentPage = totalPages;

      const start = (this.currentPage - 1) * this.perPage;
      const end = start + this.perPage;
      const pageRows = this.filteredRows.slice(start, end);

      // Render rows
      this.tbody.innerHTML = '';
      if (pageRows.length === 0) {
        const tr = document.createElement('tr');
        const td = document.createElement('td');
        td.colSpan = this.table.querySelectorAll('thead th').length;
        td.className = 'text-center ga-text-muted';
        td.style.padding = '32px';
        td.textContent = 'No data found';
        tr.appendChild(td);
        this.tbody.appendChild(tr);
      } else {
        pageRows.forEach(row => this.tbody.appendChild(row));
      }

      // Render pagination
      const footer = this.wrapper.querySelector('.ga-dt-footer');
      if (footer) {
        const info = footer.querySelector('.ga-dt-info');
        const pag = footer.querySelector('.ga-dt-pagination');

        if (info) {
          info.textContent = total === 0
            ? 'No entries'
            : `Showing ${start + 1} to ${Math.min(end, total)} of ${total} entries`;
        }

        if (pag) {
          let html = `<button ${this.currentPage <= 1 ? 'disabled' : ''} data-page="${this.currentPage - 1}">&laquo; Prev</button>`;
          for (let i = 1; i <= totalPages; i++) {
            if (totalPages > 7) {
              if (i === 1 || i === totalPages || (i >= this.currentPage - 1 && i <= this.currentPage + 1)) {
                html += `<button class="${i === this.currentPage ? 'active' : ''}" data-page="${i}">${i}</button>`;
              } else if (i === this.currentPage - 2 || i === this.currentPage + 2) {
                html += `<button disabled>...</button>`;
              }
            } else {
              html += `<button class="${i === this.currentPage ? 'active' : ''}" data-page="${i}">${i}</button>`;
            }
          }
          html += `<button ${this.currentPage >= totalPages ? 'disabled' : ''} data-page="${this.currentPage + 1}">Next &raquo;</button>`;
          pag.innerHTML = html;

          pag.querySelectorAll('button[data-page]').forEach(btn => {
            btn.addEventListener('click', () => {
              const p = parseInt(btn.dataset.page);
              if (p >= 1 && p <= totalPages) {
                this.currentPage = p;
                this.render();
              }
            });
          });
        }
      }
    }
  };

  // Tab Manager
  const TabManager = {
    init() {
      document.querySelectorAll('.ga-tabs').forEach(tabBar => {
        tabBar.querySelectorAll('.ga-tab').forEach(tab => {
          tab.addEventListener('click', () => {
            const target = tab.dataset.tab;
            if (!target) return;
            tabBar.querySelectorAll('.ga-tab').forEach(t => t.classList.remove('active'));
            tab.classList.add('active');
            const container = tabBar.parentElement;
            container.querySelectorAll('.ga-tab-content').forEach(c => c.classList.remove('active'));
            const panel = container.querySelector(`[data-tab-content="${target}"]`);
            if (panel) panel.classList.add('active');
          });
        });
      });
    }
  };

  // Toast notification helper
  window.GaToast = function(title, message, type) {
    let container = document.querySelector('.ga-toast-container');
    if (!container) {
      container = document.createElement('div');
      container.className = 'ga-toast-container';
      document.body.appendChild(container);
    }
    const colors = { success: '#10b981', danger: '#ef4444', warning: '#f59e0b', info: '#3b82f6' };
    const toast = document.createElement('div');
    toast.className = 'ga-toast';
    toast.innerHTML = `
      <svg class="ga-toast-icon" viewBox="0 0 24 24" fill="none" stroke="${colors[type]||colors.info}" stroke-width="2">
        ${type === 'success' ? '<path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/><polyline points="22 4 12 14.01 9 11.01"/>' :
          type === 'danger' ? '<circle cx="12" cy="12" r="10"/><line x1="15" y1="9" x2="9" y2="15"/><line x1="9" y1="9" x2="15" y2="15"/>' :
          type === 'warning' ? '<path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"/><line x1="12" y1="9" x2="12" y2="13"/><line x1="12" y1="17" x2="12.01" y2="17"/>' :
          '<circle cx="12" cy="12" r="10"/><line x1="12" y1="16" x2="12" y2="12"/><line x1="12" y1="8" x2="12.01" y2="8"/>'}
      </svg>
      <div class="ga-toast-body">
        <div class="ga-toast-title">${title}</div>
        <div class="ga-toast-message">${message}</div>
      </div>
      <button onclick="this.parentElement.remove()" style="background:none;border:none;cursor:pointer;color:var(--ga-text-muted);padding:0;font-size:1.2rem;">&times;</button>`;
    container.appendChild(toast);
    setTimeout(() => { toast.style.opacity = '0'; toast.style.transition = 'opacity 0.3s'; setTimeout(() => toast.remove(), 300); }, 5000);
  };

  // Init
  document.addEventListener('DOMContentLoaded', () => {
    SidebarManager.init();
    ThemeManager.init();
    DropdownManager.init();
    HeaderSearchManager.init();
    FAQManager.init();
    AlertManager.init();
    TabManager.init();
  });
})();
